/*
 * SPDX license identifier: MPL-2.0
 *
 * Copyright (C) 2015  Intel Corporation
 *
 * This file is part of GENIVI Project DLT - Diagnostic Log and Trace.
 *
 * This Source Code Form is subject to the terms of the
 * Mozilla Public License (MPL), v. 2.0.
 * If a copy of the MPL was not distributed with this file,
 * You can obtain one at http://mozilla.org/MPL/2.0/.
 *
 * For further information see http://www.genivi.org/.
 */

/*!
 * \author Stefan Vacek <stefan.vacek@intel.com> Intel Corporation
 *
 * \copyright Copyright © 2015 Intel Corporation. \n
 * License MPL-2.0: Mozilla Public License version 2.0 http://mozilla.org/MPL/2.0/.
 *
 * \file dlt-test-fork-handler.c
 */

#include <unistd.h> /* for fork() */
#include <time.h>
#include <errno.h>

#include "dlt.h"

#define SEGMENTED_MSG_SIZE (1024 * 5)


/**
 * Print usage information of tool.
 */
void usage()
{
    char version[255];

    dlt_get_version(version,255);

    printf("Usage: dlt-test-fork-handler [options]\n");
    printf("Test application executing fork handler tests\n");
    printf("%s \n", version);
    printf("Options:\n");
    printf(" - 1            Execute test 1 (test fork handling - receiver thread only)\n");
#ifdef DLT_NETWORK_TRACE_ENABLE
    printf(" - 2            Execute test 2 (test fork handling - segmented thread)\n");
#endif
}


/**
 * @brief sample code for using at_fork-handler
 */
int main(int argc, char* argv[])
{
    DltContext mainContext;
    struct timespec timeout, r;

    timeout.tv_sec  = 0;
    timeout.tv_nsec = 200000000L;
    int c;
#ifdef DLT_NETWORK_TRACE_ENABLE
    char buffer[SEGMENTED_MSG_SIZE];
    int num;
    int testSegmentedThread = 0;
#endif

    if (argc != 2)
    {
        usage();
        return 0;
    }

    c = (int) argv[1][0];
    switch (c)
    {
        case '1':
        {
            break;
        }
#ifdef DLT_NETWORK_TRACE_ENABLE
        case '2':
        {
            testSegmentedThread = 1;
            break;
        }
#endif
        case '?':
        {
            usage();
            return 0;
        }
        default:
        {
            printf("Invalid option");
            usage();
            return 0;
        }
    }

#ifdef DLT_NETWORK_TRACE_ENABLE
    if (testSegmentedThread == 1)
    {
        /* Fill up dummy buffer */
        for(num = 0; num < SEGMENTED_MSG_SIZE; num++)
        {
            buffer[num] = num;
        }
    }
#endif

    DLT_REGISTER_APP("PRNT", "Parent application");
    DLT_REGISTER_CONTEXT(mainContext, "CTXP", "Parent context");
    DLT_LOG(mainContext, DLT_LOG_WARN, DLT_STRING("First message before fork"));
    nanosleep(&timeout, &r);

#ifdef DLT_NETWORK_TRACE_ENABLE
    if (testSegmentedThread == 1)
    {
        /** Dummy messages: 16 byte header, 5k payload.
         *  Send message to ensure Segmented thread is started */
        dlt_user_trace_network_segmented(&mainContext, DLT_NW_TRACE_IPC, 16, buffer, SEGMENTED_MSG_SIZE, buffer);
        DLT_LOG(mainContext, DLT_LOG_WARN, DLT_STRING("Segmented message from parent"));
    }
#endif

    pid_t pid = fork();
    if (pid == 0) { /* child process */
        /* this message should not be visible */
        DLT_LOG(mainContext, DLT_LOG_WARN, DLT_STRING("Child's first message after fork, pid: "), DLT_INT32(getpid()));

        /* this will not register CHLD application */
        DLT_REGISTER_APP("CHLD", "Child application");
        /* this will not register CTXC context */
        DLT_REGISTER_CONTEXT(mainContext, "CTXC", "Child context");
        /* this will not log a message */
        DLT_LOG(mainContext, DLT_LOG_WARN, DLT_STRING("Child's second message after fork, pid: "), DLT_INT32(getpid()));

#ifdef DLT_NETWORK_TRACE_ENABLE
        if (testSegmentedThread == 1)
        {
            dlt_user_trace_network_segmented(&mainContext, DLT_NW_TRACE_IPC, 16, buffer, SEGMENTED_MSG_SIZE, buffer);
            DLT_LOG(mainContext, DLT_LOG_WARN, DLT_STRING("Segmented message from child"));
        }
#endif

        nanosleep(&timeout, &r);
        if (execlp("dlt-example-user", "dlt-example-user", "-n 1",
                   "you should see this message", NULL))
            return errno;
    }
    else if (pid == -1) /* error in fork */
    {
        return -1;
    }
    else { /* parent */
        DLT_LOG(mainContext, DLT_LOG_WARN, DLT_STRING("Parent's first message after fork, pid: "), DLT_INT32(getpid()));

#ifdef DLT_NETWORK_TRACE_ENABLE
        if (testSegmentedThread == 1)
        {
            dlt_user_trace_network_segmented(&mainContext, DLT_NW_TRACE_IPC, 16, buffer, SEGMENTED_MSG_SIZE, buffer);
            DLT_LOG(mainContext, DLT_LOG_WARN, DLT_STRING("Parent's first segmented message after fork, pid: "), DLT_INT32(getpid()));
        }
#endif

        nanosleep(&timeout, &r);
    }

    DLT_UNREGISTER_APP();

    return 0;
}
